import 'dotenv/config';
import express from 'express';
import { z } from 'zod';
import { McpServer } from '@modelcontextprotocol/sdk/server/mcp.js';
import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const PORT = parseInt(process.env.PORT || '2092', 10);
const app = express();
app.use(express.json());
app.use(express.static(path.join(__dirname, '..', 'public')));

const server = new McpServer({ name: 'marketing-mcp-server-ui', version: '1.1.0' });

// helper to load widget html
function loadWidget(name: string) {
  const p = path.join(__dirname, '..', 'public', 'widgets', name);
  return fs.readFileSync(p, 'utf8');
}

// Register UI resources
server.registerResource(
  'marketing-brief-widget',
  'ui://widget/marketing-brief.html',
  {},
  async () => ({
    contents: [{
      uri: 'ui://widget/marketing-brief.html',
      mimeType: 'text/html',
      text: loadWidget('marketing-brief.html')
    }]
  })
);

server.registerResource(
  'leadmagnet-widget',
  'ui://widget/leadmagnet.html',
  {},
  async () => ({
    contents: [{
      uri: 'ui://widget/leadmagnet.html',
      mimeType: 'text/html',
      text: loadWidget('leadmagnet.html')
    }]
  })
);

// Schemas
const BriefInput = z.object({
  product: z.string(),
  audience: z.string(),
  goal: z.string(),
  tone: z.string().default('informatif')
});

const LeadMagnetInput = z.object({
  topic: z.string(),
  audience: z.string()
});

// Tools bound to widgets via _meta.openai/outputTemplate
server.registerTool(
  'generate_marketing_brief',
  {
    title: 'Generate Marketing Brief',
    description: 'Membuat brief iklan ringkas dari produk, audiens, dan tujuan kampanye',
    _meta: { 'openai/outputTemplate': 'ui://widget/marketing-brief.html' },
    inputSchema: {
      product: { type: 'string' },
      audience: { type: 'string' },
      goal: { type: 'string' },
      tone: { type: 'string' }
    }
  },
  async (args) => {
    const { product, audience, goal, tone } = BriefInput.parse(args);
    const brief = {
      product,
      audience,
      goal,
      messaging: [
        'Nilai unik yang relevan untuk audiens',
        'Ajakan bertindak yang jelas dan spesifik'
      ],
      tone
    };
    return {
      structuredContent: brief,
      content: [{ type: 'text', text: 'Brief iklan siap ditinjau di widget.' }]
    };
  }
);

server.registerTool(
  'generate_lead_magnet_outline',
  {
    title: 'Generate Lead Magnet Outline',
    description: 'Menyusun outline lead magnet berdasar topik dan audiens',
    _meta: { 'openai/outputTemplate': 'ui://widget/leadmagnet.html' },
    inputSchema: {
      topic: { type: 'string' },
      audience: { type: 'string' }
    }
  },
  async (args) => {
    const { topic, audience } = LeadMagnetInput.parse(args);
    const outline = {
      title: `Lead Magnet tentang ${topic}`,
      audience,
      sections: [
        { heading: 'Masalah Utama', points: ['Nyeri pengguna', 'Dampak bisnis'] },
        { heading: 'Solusi Inti', points: ['Kerangka solusi', 'Contoh cepat'] },
        { heading: 'Ajakan Bertindak', points: ['Benefit utama', 'Link pendaftaran'] }
      ]
    };
    return {
      structuredContent: outline,
      content: [{ type: 'text', text: 'Outline lead magnet berhasil dibuat, cek widget.' }]
    };
  }
);

// MCP endpoint
app.post('/mcp', async (req, res) => {
  try {
    const reply = await server.handleRequest(req.body);
    res.json(reply);
  } catch (err: any) {
    console.error('MCP error:', err);
    res.status(500).json({ error: err?.message || 'Internal error' });
  }
});

app.listen(PORT, () => {
  console.log(`MCP server listening on http://localhost:${PORT}/mcp`);
});
