import 'dotenv/config';
import express from 'express';
import { z } from 'zod';
import { McpServer } from '@modelcontextprotocol/sdk/server/mcp.js';

const PORT = parseInt(process.env.PORT || '2091', 10);
const app = express();
app.use(express.json());

// Inisialisasi MCP Server
const server = new McpServer({
  name: 'marketing-mcp-server',
  version: '1.0.0'
});

// Skema input output
const BriefInput = z.object({
  product: z.string(),
  audience: z.string(),
  goal: z.string(),
  tone: z.string().default('informatif')
});

const LeadMagnetInput = z.object({
  topic: z.string(),
  audience: z.string()
});

// Tool 1: Generate Marketing Brief
server.registerTool(
  'generate_marketing_brief',
  {
    title: 'Generate Marketing Brief',
    description: 'Membuat brief iklan ringkas dari produk, audiens, dan tujuan kampanye',
    inputSchema: {
      product: { type: 'string' },
      audience: { type: 'string' },
      goal: { type: 'string' },
      tone: { type: 'string' }
    }
  },
  async (args) => {
    const { product, audience, goal, tone } = BriefInput.parse(args);
    const brief = {
      product,
      audience,
      goal,
      messaging: [
        'Nilai unik yang relevan untuk audiens',
        'Ajakan bertindak yang jelas dan spesifik'
      ],
      tone
    };
    return {
      // structuredContent memudahkan UI/komponen merender data ini
      structuredContent: brief,
      content: [{ type: 'text', text: 'Brief iklan siap ditinjau.' }]
    };
  }
);

// Tool 2: Generate Lead Magnet Outline
server.registerTool(
  'generate_lead_magnet_outline',
  {
    title: 'Generate Lead Magnet Outline',
    description: 'Menyusun outline lead magnet berdasar topik dan audiens',
    inputSchema: {
      topic: { type: 'string' },
      audience: { type: 'string' }
    }
  },
  async (args) => {
    const { topic, audience } = LeadMagnetInput.parse(args);
    const outline = {
      title: `Lead Magnet tentang ${topic}`,
      audience,
      sections: [
        { heading: 'Masalah Utama', points: ['Nyeri pengguna', 'Dampak bisnis'] },
        { heading: 'Solusi Inti', points: ['Kerangka solusi', 'Contoh cepat'] },
        { heading: 'Ajakan Bertindak', points: ['Benefit utama', 'Link pendaftaran'] }
      ]
    };
    return {
      structuredContent: outline,
      content: [{ type: 'text', text: 'Outline lead magnet berhasil dibuat.' }]
    };
  }
);

// Endpoint JSON-RPC MCP
app.post('/mcp', async (req, res) => {
  try {
    const reply = await server.handleRequest(req.body);
    res.json(reply);
  } catch (err: any) {
    console.error('MCP error:', err);
    res.status(500).json({ error: err?.message || 'Internal error' });
  }
});

app.get('/', (_req, res) => {
  res.send('Marketing MCP Server is running. POST JSON-RPC to /mcp');
});

app.listen(PORT, () => {
  console.log(`MCP server listening on http://localhost:${PORT}/mcp`);
});
